/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier MN2WS0230 PCM Driver");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE(
	"{{Panasonic, MN2WS0230}, "
	"{Panasonic, MN2WS0230x}}");


#define MN2WS_PCM_DEV_NAME       "mn2ws0230-pcm"
#define MN2WS_PCM_DEV_COUNT      2

#define MN2WS_RES_PCM            0

#define MN2WS_PCM_REG_START      (0x5e00c000)
#define MN2WS_PCM_REG_SIZE       (0x00006000)

#define MN2WS_PCM_HW_BUF_SIZE    (PAGE_SIZE * 2)


//10c00
#define PCM_ADECOUTCTL        (0x5e010c00)

//10f10, 10f18
#define PCM_PCMOCTL_0         (0x5e010f10)
#define PCM_IECOCTL           (0x5e010f18)

//fd0c, ff0c, 1010c, 1030c
#define PCM_AUDCTL0(ch) \
	((ch == 0) ? 0x5e00fd0c : \
	(ch == 1) ? 0x5e00ff0c : \
	(ch == 2) ? 0x5e01010c : \
	(ch == 3) ? 0x5e01030c : \
	0)

//105cc, 107cc
#define PCM_EXTPCMPLAYCTL(ch) \
	((ch == 0) ? 0x5e0105cc : \
	(ch == 1) ? 0x5e0107cc : \
	0)
#define PCM_EXTPCMBUFSTADDR(ch) \
	((ch == 0) ? 0x5e0105d0 : \
	(ch == 1) ? 0x5e0107d0 : \
	0)
#define PCM_EXTPCMBUFENDADDR(ch) \
	((ch == 0) ? 0x5e0105d4 : \
	(ch == 1) ? 0x5e0107d4 : \
	0)

#define PLAYCTL_PLAYSW      ( 1 <<  0) //1'b1
#define PLAYCTL_FSOUTSEL    ( 1 <<  3) //1'b1
#define PLAYCTL_FS          ( 7 <<  4) //3'b111
#define PLAYCTL_SCALE       (15 <<  8) //4'b1111
#define PLAYCTL_DECSCALE    (15 << 16) //4'b1111


//105d8, 107d8
#define PCM_EXTPCMBUFTHSIZE(ch) \
	((ch == 0) ? 0x5e0105d8 : \
	(ch == 1) ? 0x5e0107d8 : \
	0)
#define PCM_EXTPCMBUFWRPTR(ch) \
	((ch == 0) ? 0x5e0105dc : \
	(ch == 1) ? 0x5e0107dc : \
	0)
#define PCM_EXTPCMBUFRDPTR(ch) \
	((ch == 0) ? 0x5e0105e0 : \
	(ch == 1) ? 0x5e0107e0 : \
	0)


static int mn2ws0230_pcm_hwdep_force_aout(struct mn2ws_pcm_dev *dev);

static int mn2ws0230_pcm_play_setup(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_play_mute(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_play_start(struct mn2ws_pcm_dev *dev);
static int mn2ws0230_pcm_play_stop(struct mn2ws_pcm_dev *dev);
static unsigned long mn2ws0230_pcm_play_get_hwptr(struct mn2ws_pcm_dev *dev);
static void mn2ws0230_pcm_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr);
static int mn2ws0230_pcm_init_module(void);
static void mn2ws0230_pcm_exit_module(void);


static struct resource mn2ws_resources[] = {
	[MN2WS_RES_PCM] = {
		.start = MN2WS_PCM_REG_START, 
		.end   = MN2WS_PCM_REG_START + MN2WS_PCM_REG_SIZE, 
		.flags = IORESOURCE_MEM, 
	}, 
};

struct mn2ws_pcm_desc mn2ws_pcm[] = {
	//ch 0
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 1, 
			.force_aout    = mn2ws0230_pcm_hwdep_force_aout, 
		}, 
		.play = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			//18.6ms(min: 1536bytes = 8ms)
			.size_use_fix  = 256 * 14, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_48k_16b_2ch, 
			.setup         = mn2ws0230_pcm_play_setup, 
			.mute          = mn2ws0230_pcm_play_mute, 
			.start         = mn2ws0230_pcm_play_start, 
			.stop          = mn2ws0230_pcm_play_stop, 
			.get_hwptr     = mn2ws0230_pcm_play_get_hwptr, 
			.set_devptr    = mn2ws0230_pcm_play_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_play_copy_to_hw, 
			.silence       = mn2ws_pcm_play_silence_to_hw, 
		}, 
	}, 
	
	//ch 1
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 0, 
		}, 
		.play = {
			//ch 1 is not supported by HW...
			.enabled       = 0, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			//18.6ms(min: 1536bytes = 8ms)
			.size_use_fix  = 256 * 14, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_48k_16b_2ch, 
			.setup         = mn2ws0230_pcm_play_setup, 
			.mute          = mn2ws0230_pcm_play_mute, 
			.start         = mn2ws0230_pcm_play_start, 
			.stop          = mn2ws0230_pcm_play_stop, 
			.get_hwptr     = mn2ws0230_pcm_play_get_hwptr, 
			.set_devptr    = mn2ws0230_pcm_play_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_play_copy_to_hw, 
			.silence       = mn2ws_pcm_play_silence_to_hw, 
		}, 
	}, 
};


const char *get_pcm_dev_name(void)
{
	return MN2WS_PCM_DEV_NAME;
}

int get_pcm_dev_count(void)
{
	return MN2WS_PCM_DEV_COUNT;
}

static int mn2ws0230_pcm_hwdep_force_aout(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	
	//settings: Main:DEC0, IEC:DEC1
	
	//0    : enable : off
	pcm_writel(mmp, 0x0000c020, PCM_PCMOCTL_0);
	pcm_writel(mmp, 0x00110002, PCM_IECOCTL);
	
	//0    : enable : on
	pcm_writel(mmp, 0x0000c021, PCM_PCMOCTL_0);
	pcm_writel(mmp, 0x00110003, PCM_IECOCTL);
	
	//2    : output DEC2
	//1    : output DEC1
	//0    : output DEC0
	pcm_writel(mmp, 0x00000001, PCM_ADECOUTCTL);
	
	//20-23: init fs    : 44.1kHz
	//18   : IEC copy   : can copy
	//17   : IEC output : PCM
	//pcm_writel(mmp, 0x00040000, PCM_AUDCTL0(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_play_setup(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 st, ed;
	u32 v;
	
	//DMA buffer address
	st = dev->play.paddr_buf - 0x80000000;
	ed = st + dev->play.size_buf_use;
	pcm_writel(mmp, st, PCM_EXTPCMBUFSTADDR(dev->ch));
	pcm_writel(mmp, ed, PCM_EXTPCMBUFENDADDR(dev->ch));
	pcm_writel(mmp, st, PCM_EXTPCMBUFWRPTR(dev->ch));
	
	//DMA start immediately
	pcm_writel(mmp, 8, PCM_EXTPCMBUFTHSIZE(dev->ch));
	
	//Initialize the sound mode
	//16-19: Scaler decoder     : not set
	// 8-11: Scaler LPCM        : not set
	// 4- 6: 48kHz 2ch
	// 3   : Mix decoder output : not set
	// 0   : Start DMA          : not set
	v = 0;
	//v |= PLAYCTL_PLAYSW;
	//v |= PLAYCTL_FSOUTSEL;
	v |= PLAYCTL_FS;
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	//Set scaler
	mn2ws0230_pcm_play_mute(dev);
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_play_mute(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	//Set the scaler mode
	//16-19: Scaler decoder
	// 8-11: Scaler LPCM
	v = pcm_readl(mmp, PCM_EXTPCMPLAYCTL(dev->ch));
	v &= ~PLAYCTL_DECSCALE;
	v |= (dev->hwd.scale_dec << 16);
	v &= ~PLAYCTL_SCALE;
	v |= (dev->hwd.scale_pcm <<  8);
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_play_start(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	DPRINTF("start the playback DMA %d.\n", dev->ch);
	
	//start the DMA
	v = pcm_readl(mmp, PCM_EXTPCMPLAYCTL(dev->ch));
	v |= PLAYCTL_PLAYSW;
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0230_pcm_play_stop(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	DPRINTF("stop the playback DMA %d.\n", dev->ch);
	
	//stop the DMA
	v = pcm_readl(mmp, PCM_EXTPCMPLAYCTL(dev->ch));
	v &= ~PLAYCTL_PLAYSW;
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	return 0;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤ HW ɤ߼֤롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @return HW θߤɤ߼
 */
static unsigned long mn2ws0230_pcm_play_get_hwptr(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	v = pcm_readl(mmp, PCM_EXTPCMBUFRDPTR(dev->ch)) - 
		pcm_readl(mmp, PCM_EXTPCMBUFSTADDR(dev->ch));
	
	return v;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤΥǥХν񤭹֤߰ꤹ롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @param ptr ǥХθߤν񤭹߰
 */
static void mn2ws0230_pcm_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 wr, wr_new;
	
	wr = pcm_readl(mmp, PCM_EXTPCMBUFWRPTR(dev->ch));
	wr_new = pcm_readl(mmp, PCM_EXTPCMBUFSTADDR(dev->ch)) + ptr;
	
	if (wr != wr_new) {
		pcm_writel(mmp, wr_new, PCM_EXTPCMBUFWRPTR(dev->ch));
	}
}


static int __init mn2ws0230_pcm_init_module(void)
{
	return mn2ws_pcm_init_module(mn2ws_pcm);
}

static void __exit mn2ws0230_pcm_exit_module(void)
{
	mn2ws_pcm_exit_module(mn2ws_pcm);
}

module_init(mn2ws0230_pcm_init_module);
module_exit(mn2ws0230_pcm_exit_module);
